define([
    'underscore',
    'backbone',
    'Layout',
    'modules/new-appointment-request/views/direct/clinics/single-clinic-view',
    'modules/new-appointment-request/resources/direct/custom-messages/collection',
    'modules/new-appointment-request/resources/direct/clinics/collection',
    'modules/new-appointment-request/tests/helpers/general-helper',
], function(_, Backbone, Layout, SingleClinicView, CustomMessagesCollection, DataCollection, helpers) {
    'use strict';

    describe('Single Clinic View', function() {
        var view;
        var Model = DataCollection.prototype.model;
        var clinicsTimeSlotsModel;
        var $view;
        var model;
        var customMessages;
        var layout;

        beforeEach(function() {

            model = new Backbone.Model({
                friendlyName: 'friendlyBoston',
                facility: new Backbone.Model({
                    'institutionCode': '12345',
                }),
            });
            model.getSiteCode = function() {
                return '123';
            };

            layout = new Layout();
            layout.render();

            customMessages = new CustomMessagesCollection([
                {
                    'id': 'noClinicRadioButtons',
                    'messageId': 'noClinicRadioButtons',
                    'messageText': 'This is a test.',
                },
            ]);
        });


        describe('with appointment time slots', function() {
            beforeEach(function() {
                clinicsTimeSlotsModel = new Model({
                    clinicName: 'Boston',
                    clinicFriendlyLocationName: 'friendlyBoston',
                    appointmentTimeSlot: [1],
                });

                view = new SingleClinicView({
                    model: model,
                    clinicsTimeSlotsModel: clinicsTimeSlotsModel,
                    clinicCustomMessage: customMessages.get('noClinicRadioButtons'),
                });

                layout.showChildView('content', view);
                $view = view.$el;
            });

            it('shows custom message text', function() {
                expect($view.find('.custom-message-instructions').text()).toEqual('This is a test.');
            });

            it('shows clinic name, friendly name, and available slots', function() {
                expect($view.find('.static-label').text()).toEqual('Clinic:');
                expect(helpers.cleanString($view.find('.label-text > span').text())).toEqual('Boston friendlyBoston');
            });
        });

        describe('without any appointment time slots', function() {
            beforeEach(function() {
                clinicsTimeSlotsModel = new Model({
                    clinicName: 'Boston',
                    clinicFriendlyLocationName: 'friendlyBoston',
                    appointmentTimeSlot: [],
                });

                view = new SingleClinicView({
                    model: model,
                    clinicsTimeSlotsModel: clinicsTimeSlotsModel,
                    clinicCustomMessage: customMessages.get('noClinicRadioButtons'),
                });

                layout.showChildView('content', view);
                $view = view.$el;
            });

            it('shows custom message text', function() {
                expect($view.find('.custom-message-instructions').text()).toEqual('This is a test.');
            });

            it('shows clinic name, friendly name, and informational text', function() {
                expect($view.find('.static-label').text()).toEqual('Clinic:');
                expect(helpers.cleanString($view.find('.label-text > span').text())).toEqual('Boston friendlyBoston');
            });
        });

        describe('getTimeSlots', function() {
            beforeEach(function() {
                spyOn(SingleClinicView.prototype, 'checkClinic').and.callFake(_.noop);
                clinicsTimeSlotsModel = new Model({
                    clinicName: 'Boston',
                    clinicFriendlyLocationName: 'friendlyBoston',
                    appointmentTimeSlot: [1],
                });

                view = new SingleClinicView({
                    model: model,
                    clinicsTimeSlotsModel: clinicsTimeSlotsModel,
                });
                layout.showChildView('content', view);
            });

            it('returns null when there are no slots', function() {
                view.slots = undefined;
                expect(view.getTimeSlots(null)).toBeNull();
            });

            it('returns the time slot when it exists', function() {
                var ID = '123';
                var m = new Backbone.Model({id: ID, data: '123'});

                view.slots = new Backbone.Collection([m]);

                expect(view.getTimeSlots(ID).get('data')).toBe('123');
            });
        });

        describe('fetchSlotsSuccess', function() {
            beforeEach(function() {
                spyOn(SingleClinicView.prototype, 'checkClinic').and.callFake(_.noop);
                clinicsTimeSlotsModel = new Model({
                    clinicName: 'Boston',
                    clinicFriendlyLocationName: 'friendlyBoston',
                    appointmentTimeSlot: [1],
                });

                view = new SingleClinicView({
                    model: model,
                    clinicsTimeSlotsModel: clinicsTimeSlotsModel,
                });

                spyOn(view, 'showErrorMessage').and.callFake(_.noop);
                layout.showChildView('content', view);
            });

            it('shows an error when slots are missing', function() {
                view.fetchSlotsSuccess(new Backbone.Collection());
                expect(view.showErrorMessage).toHaveBeenCalled();
            });

            it('sets a clinic when successful', function() {
                var m = new Backbone.Model({id: '123', data: 'abc', appointmentTimeSlot: ['1']});
                var c = new Backbone.Collection([m]);
                view.clinicList = new Backbone.Collection([m]);
                view.fetchSlotsSuccess(c);

                expect(view.showErrorMessage).not.toHaveBeenCalled();
                expect(view.model.get('clinic').toJSON()).toEqual(clinicsTimeSlotsModel.toJSON());
            });
        });
    });
}
);
